﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;
using System.IO;
//#if M9393
//using Vsag = NSM9393A.Vsag_M9393A;
//#else
//using Vsag = NSM9391A.Vsag_M9391A;
//#endif

namespace PowerAmpDemoNS
{
    public partial class PowerAmpDemoProgram
    {
        // Public properties that are accessed directly in the GUI
        public DataLog dataLog;
        public IVsag vsag;
        public DcSmu dcSmu;
        public Dio dio;
        public Arb etArb;

        public double[] cellFreqs;
        public double[] wlanFreqs;

        public int numCellHarms = 6;
        public int numWlanHarms = 1;

        public double targetPout = 6;
        public double targetGain = 38;
        public double poutMargin = 0.05;

        #region Waveform Names
        // RF Waveform Files
        private string gsmArb;
        private string edgeArb;
        private string evDoArb;
        private string wcdmaArb;
        private string tdscdmaArb;
        private string cdma2000Arb;
        private string lte1_4MhzArb;
        private string lte5MhzArb;
        private string lte10MhzArb;
        private string lte20MhzArb;
        private string lteTdd5MhzArb;
        private string lteTdd10MhzArb;
        private string lteTdd20MhzArb;
        private string WlanN20MhzArb;
        private string WlanN40MhzArb;
        private string WlanAC40MhzArb;
        private string WlanAC80MhzArb;
        private string WlanAC160MhzArb;
        private string cwArb;
        private string powerRampArb;

        // ET Waveform Names
        public string lte1_4MhzEtArb;
        public string lte5MhzEtArb;
        public string lte10MhzEtArb;
        public string lte20MhzEtArb;
        #endregion

        #region Test Parameters
        // ET Parameters
        public double iqDelay = 200e-9;
        public double iqDelay5MHz = 223e-9;
        public double iqDelay10MHz = 193e-9;
        public double iqDelay20MHz = 182e-9;
        public double iqDelayOffset = 0;
        public double iqRange = 10e-9;
        public double iqStep = 1e-9;
        public int frequencyIndex = 0;
        public double arbAmplitude = 0.5;
        public double arbOffset = 0.2;
        public bool enableEt = false;
        public double vRef = 2.75;
        public double etpsGain = 4.0;
        public double etWaveformLength = 0.01;

        // WLAN Dynamic EVM Settings
        public double dutyCycle = 0.5;
        public double dcLeadTime = 10e-6;
        public double dcLagTime = 10e-6;
        public bool useDynamicEvm = false;
        public double pulseVoltage = 2.5;
        public int evmAvgCount = 1;
        public int semAvgCount = 5;
        #endregion

        // Enable Measurements
        private bool testCurr = true;
        private bool testAcpr = true;
        private bool testEvm = true;
        private bool testHarms = true;
        private bool testSem = true;
        public bool testEtSweep = true;
        public bool testCompression = true;

        public bool dcOffAfterTests = false;
        private Stopwatch sw = new Stopwatch();

        // Allocate space for test results
        private int numTests = 500;
        private int maxAcprMeas = 8;
        private int numCurrMeas = 10;

        public PowerAmpDemoProgram()
        {
            dataLog = new DataLog(numTests, maxAcprMeas, numCurrMeas);

            dio = new Dio();
            etArb = new Arb();
            dcSmu = new DcSmu(dataLog);

            // Set any default parameter values
            targetPout = -5;
            targetGain = 16;
            poutMargin = 0.05;

            initTestFreqs();
        }

        #region Initialize GUI

        public void configLists(ListBox testListBox, ListBox measListBox, ListBox xAppListBox)
        {
            string[] testList = { "CW", "GSM", "EDGE", "WCDMA", "CDMA2000", "EVDO", "TD-SCDMA", 
                                    "LTE 5 MHz", "LTE 10 MHz", "LTE 20 MHz",  
                                    "LTE-TDD 5 MHz", "LTE-TDD 10 MHz", "LTE-TDD 20 MHz", 
                                    "802.11N 20 MHz", "802.11N 40 MHz", "802.11AC 40 MHz", "802.11AC 80 MHz" };
            foreach (string testName in testList)
            {
                testListBox.Items.Add(testName);
            }

            string[] measList = { "Power Servo Only", "DC Current", "ACPR", "EVM", "Harmonics", "SEM", "ET ACPR Sweep", "Compression" };
            for (int i = 0; i < measList.Count(); i++)
            {
                measListBox.Items.Add(measList[i]);
            }

            string[] xAppList = { "EDGEGSM", "CDMA2K", "CDMA1XEV", "WCDMA", "TDSCDMA", "LTE", "LTETDD", "WLAN" };
            // Skip LTE-TDD and TTD-SCDMA until startup issue is resolved
            //string[] xAppList = { "EDGEGSM", "CDMA2K", "CDMA1XEV", "WCDMA", "LTE", "WLAN" };
            for (int i = 0; i < xAppList.Count(); i++)
            {
                xAppListBox.Items.Add(xAppList[i]);
            }
        }
        #endregion

        public void createVsag(string vsaType)
        {
            switch (vsaType)
            {
                case "M9393A":
                    vsag = new NSM9393A.Vsag_M9393A(numTests, dataLog);
                    break;
                case "M9391A":
                    vsag = new NSM9391A.Vsag_M9391A(numTests, dataLog);
                    break;
                default:
                    throw new Exception("Unsupported VSA Type:  " + vsaType);
            }
            // Load the calibration data
            vsag.initCalData(1, 6);
        }



        public void initializeInstruments(string[] xAppsToLoad)
        {
            vsag.initVsgVsa();
            vsag.loadXapps(xAppsToLoad);
            vsag.configureTriggers();

            etArb.initEtArb();
            dcSmu.initDcSmus();
            dio.initRffeDio();

            #region Load Waveforms

            // Read waveforms from file
            // Open the file
            var rs = new FileStream(Application.StartupPath + "\\" + "Waveforms.csv", FileMode.Open);
            TextReader rd = new StreamReader(rs);
            //bool reading = true;
            while (rd.Peek() > 0)
            {
                try
                {
                    string fileString = rd.ReadLine();
                    //fileString = fileString.ToUpper();
                    string[] columns = fileString.Split(',');
                    switch (columns[0])
                    {
                        case "gsmArb":
                            gsmArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), gsmArb);
                            break;
                        case "edgeArb":
                            edgeArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), edgeArb);
                            break;
                        case "evDoArb":
                            evDoArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), evDoArb);
                            break;
                        case "wcdmaArb":
                            wcdmaArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), wcdmaArb);
                            break;
                        case "tdscdmaArb":
                            tdscdmaArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), tdscdmaArb);
                            break;
                        case "cdma2000Arb":
                            cdma2000Arb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), cdma2000Arb);
                            break;
                        case "lte1_4MhzArb":
                            lte1_4MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lte1_4MhzArb);
                            break;
                        case "lte5MhzArb":
                            lte5MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lte5MhzArb);
                            break;
                        case "lte10MhzArb":
                            lte10MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lte10MhzArb);
                            break;
                        case "lte20MhzArb":
                            lte20MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lte20MhzArb);
                            break;
                        case "lteTdd5MhzArb":
                            lteTdd5MhzArb = columns[2];
                            if (lteTdd5MhzArb.Contains(".WAVEFORM"))
                            {
                                vsag.loadShortVsgWaveform(Path.GetFullPath(columns[1]), lteTdd5MhzArb, 2e-3, 5e-3);
                                lteTdd5MhzArb = lteTdd5MhzArb + "Short";
                            }
                            else
                                vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lteTdd5MhzArb);
                            break;
                        case "lteTdd10MhzArb":
                            lteTdd10MhzArb = columns[2];
                            if (lteTdd10MhzArb.Contains(".WAVEFORM"))
                            {
                                vsag.loadShortVsgWaveform(Path.GetFullPath(columns[1]), lteTdd10MhzArb, 2e-3, 5e-3);
                                lteTdd10MhzArb = lteTdd10MhzArb + "Short";
                            }
                            else
                                vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lteTdd10MhzArb);
                            break;
                        case "lteTdd20MhzArb":
                            lteTdd20MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), lteTdd20MhzArb);
                            break;
                        case "WlanN20MhzArb":
                            WlanN20MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), WlanN20MhzArb);
                            break;
                        case "WlanN40MhzArb":
                            WlanN40MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), WlanN40MhzArb);
                            break;
                        case "WlanAC40MhzArb":
                            WlanAC40MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), WlanAC40MhzArb);
                            break;
                        case "WlanAC80MhzArb":
                            WlanAC80MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), WlanAC80MhzArb);
                            break;
                        case "WlanAC160MhzArb":
                            WlanAC160MhzArb = columns[2];
                            vsag.loadVsgWaveform(Path.GetFullPath(columns[1]), WlanAC160MhzArb);
                            break;

                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
            rd.Close();
            rs.Close();

            // Create a CW (DC) ARB and a ramp ARB for Power Compression Tests
            cwArb = "CW ARB";
            powerRampArb = "Power Ramp Arb";
            vsag.generatePowerRampArb(cwArb, 0, 0, 100e-6, 4e6);
            vsag.generatePowerRampArb(powerRampArb, -30, 0, 100e-6, 4e6);

            // Load ET ARB Files
            loadEtSettings();

            #endregion

        }

        public void closeInstruments()
        {
            if (vsag != null)
                vsag.closeVsgVsa();
            if (etArb != null)
                etArb.closeEtArb();
            if (dcSmu != null) 
                dcSmu.closeDcSmus();
        }

        public void loadEtSettings()
        {
            etArb.clearAwgMemory();

            bool etpsGainRead = false;
            bool vRefRead = false;
            bool etWaveformLengthRead = false;
            // Read waveforms from file
            // Open the file
            var rs = new FileStream(Application.StartupPath + "\\" + "ET Settings.csv", FileMode.Open);
            TextReader rd = new StreamReader(rs);
            //bool reading = true;
            while (rd.Peek() > 0)
            {
                try
                {
                    string fileString = rd.ReadLine();
                    //fileString = fileString.ToUpper();
                    string[] columns = fileString.Split(',');
                    switch (columns[0])
                    {
                        case "lte5MhzEtArb":
                            lte5MhzEtArb = columns[2];
                            if (etpsGainRead && vRefRead && etWaveformLengthRead)
                                etArb.loadAwgWaveform(Path.GetFullPath(columns[1]), lte5MhzEtArb, "LTE_5MHz", vRef, etpsGain, true, etWaveformLength);
                            else
                                throw new Exception("etWaveformLength, etpsGain and vRef must be read before loading ET waveforms");
                            break;
                        case "lte10MhzEtArb":
                            lte10MhzEtArb = columns[2];
                            if (etpsGainRead && vRefRead && etWaveformLengthRead)
                                etArb.loadAwgWaveform(Path.GetFullPath(columns[1]), lte10MhzEtArb, "LTE_10MHz", vRef, etpsGain, true, etWaveformLength);
                            else
                                throw new Exception("etWaveformLength, etpsGain and vRef must be read before loading ET waveforms");
                            break;
                        case "lte20MhzEtArb":
                            lte20MhzEtArb = columns[2];
                            if (etpsGainRead && vRefRead && etWaveformLengthRead)
                                etArb.loadAwgWaveform(Path.GetFullPath(columns[1]), lte20MhzEtArb, "LTE_20MHz", vRef, etpsGain, true, etWaveformLength);
                            else
                                throw new Exception("etWaveformLength, etpsGain and vRef must be read before loading ET waveforms");
                            break;
                        case "iqDelay5MHz":
                            iqDelay5MHz = Convert.ToDouble(columns[1]);
                            break;
                        case "iqDelay10MHz":
                            iqDelay10MHz = Convert.ToDouble(columns[1]);
                            break;
                        case "iqDelay20MHz":
                            iqDelay20MHz = Convert.ToDouble(columns[1]);
                            break;
                        case "arbAmplitude":
                            arbAmplitude = Convert.ToDouble(columns[1]);
                            break;
                        case "arbOffset":
                            arbOffset = Convert.ToDouble(columns[1]);
                            break;
                        case "vRef":
                            vRef = Convert.ToDouble(columns[1]);
                            vRefRead = true;
                            break;
                        case "etpsGain":
                            etpsGain = Convert.ToDouble(columns[1]);
                            etpsGainRead = true;
                            break;
                        case "etWaveformLength":
                            etWaveformLength = Convert.ToDouble(columns[1]);
                            etWaveformLengthRead = true;
                            break;
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
            lte5MhzEtArb = lte5MhzArb.Replace(".wfm", ".csv");

            // Generate a DC ramp signal for the ET input
            //Lib.generateDcRampArb(powerRampArb + ".csv", .2, 1, 100e-6, 4e6);
        }

        public int runSelectedTests(string[] testNames, string[] measNames)
        {
            int status = 0;
            //Use rfOnOff to turn the RF power off on the DUT after test or not
            //bool rfOnOff = false;
            try
            {
                // Configure the measurements to run from GUI Listbox values
                testCurr = testAcpr = testEvm = testHarms = testSem = testEtSweep = testCompression = false;
                foreach (string measName in measNames)
                {
                    if (measName == "DC Current") testCurr = true;
                    if (measName == "ACPR") testAcpr = true;
                    if (measName == "EVM") testEvm = true;
                    if (measName == "Harmonics") testHarms = true;
                    if (measName == "SEM") testSem = true;
                    if (measName == "ET ACPR Sweep") testEtSweep = true;
                    if (measName == "Compression") testCompression = true;
                }

                dataLog.testIndex = 0;
                foreach (string testName in testNames)
                {
                    if (testName == "CW") status = cwTest();
                    if (testName == "GSM") status = gsmTest();
                    if (testName == "EDGE") status = edgeTest();
                    if (testName == "WCDMA") status = wcdmaTest();
                    if (testName == "CDMA2000") status = cdma2000Test();
                    if (testName == "EVDO") status = evDoTest();
                    if (testName == "TD-SCDMA") status = tdscdmaTest();
                    if (testName == "LTE 5 MHz") status = lteTest("B5M");
                    if (testName == "LTE 10 MHz") status = lteTest("B10M");
                    if (testName == "LTE 20 MHz") status = lteTest("B20M");
                    if (testName == "LTE-TDD 5 MHz") status = lteTddTest("B5M");
                    if (testName == "LTE-TDD 10 MHz") status = lteTddTest("B10M");
                    if (testName == "LTE-TDD 20 MHz") status = lteTddTest("B20M");
                    if (testName == "802.11N 20 MHz") status = wlanTest("N20", WlanN20MhzArb, evmAvgCount, semAvgCount);
                    if (testName == "802.11N 40 MHz") status = wlanTest("N40", WlanN40MhzArb, evmAvgCount, semAvgCount);
                    if (testName == "802.11AC 40 MHz") status = wlanTest("AC40", WlanAC40MhzArb, evmAvgCount, semAvgCount);
                    if (testName == "802.11AC 80 MHz") status = wlanTest("AC80", WlanAC80MhzArb, evmAvgCount, semAvgCount);
                }
            }
            finally
            {
                dataLog.endDutLog();
            }

            return 0;
        }

        public void runPowerCalibration(string pmResource, string filePath)
        {
            double[] calFreqs = new double[cellFreqs.Length + wlanFreqs.Length];
            int indx = 0;
            for (int i = 0; i < cellFreqs.Length; i++)
            {
                calFreqs[indx] = cellFreqs[i];
                indx++;
            }
            for (int i = 0; i < wlanFreqs.Length; i++)
            {
                calFreqs[indx] = wlanFreqs[i];
                indx++;
            }

            string testWaveform = cwArb;
            vsag.setupParameters("CW");
            double calPowerlevel = 0;
            double nominalInputLoss = 0;
            double nominalOutputLoss = 10;
            vsag.runPowerCal(pmResource, testWaveform, calFreqs, calPowerlevel, filePath, true, nominalInputLoss, nominalOutputLoss);
        }

        public void initTestFreqs()
        {
            // Clean up any old cal data
            if (cellFreqs != null) cellFreqs = null;
            if (wlanFreqs != null) wlanFreqs = null;

            cellFreqs = new double[] { 1.7e9, 1.8e9, 1.9e9 };
            wlanFreqs = new double[] { 5.2e9, 5.55e9, 5.8e9 };

        }

        // readFreqData
        // Reads a test frequency data file with the following format:
        /*
            Cell Freqs,6.00E+00
            WLAN Freqs,6.00E+00
            ,
            Cell Data,
            Frequency,
            1.71E+09,
            1.75E+09,
            1.79E+09,
            1.81E+09,
            1.85E+09,
            1.88E+09,
            ,
            WLAN Data,
            Frequency,
            2.40E+09,
            2.42E+09,
            2.44E+09,
            5.20E+09,
            5.50E+09,
            5.80E+09,
        */
        public void readFreqData(string filePath, string fileName)
        {
            int numCellPoints = 0;
            int numWlanPoints = 0;



            // Clean up any old cal data
            if (cellFreqs != null) cellFreqs = null;
            if (wlanFreqs != null) wlanFreqs = null;

            // Open the file
            var rs = new FileStream(filePath + fileName, FileMode.Open);
            TextReader rd = new StreamReader(rs);

            // Read the data size values
            while (numCellPoints == 0 || numWlanPoints == 0)
            {
                string fileString = rd.ReadLine();
                fileString = fileString.ToUpper();
                string[] columns = fileString.Split(',');
                if (columns[0].Contains("CELL FREQS")) numCellPoints = Convert.ToInt32(columns[1]);
                if (columns[0].Contains("WLAN FREQS")) numWlanPoints = Convert.ToInt32(columns[1]);
            }

            // Size the Frequency arrays
            cellFreqs = new double[numCellPoints];
            wlanFreqs = new double[numWlanPoints];

            bool cellDataRead = false;
            bool wlanDataRead = false;
            while (cellDataRead == false || wlanDataRead == false)
            {
                string fileString = rd.ReadLine();
                fileString = fileString.ToUpper();
                if (fileString.Contains("CELL DATA"))
                {
                    int linesRead = 0;
                    while (linesRead < numCellPoints)
                    {
                        fileString = rd.ReadLine();
                        fileString = fileString.ToUpper();
                        if (fileString.Contains("NUMHARMS"))
                        {
                            // Read the cal data
                            string[] columns = fileString.Split(',');
                            numCellHarms = Convert.ToInt32(columns[1]);
                            continue;
                        }
                        if (fileString.Contains("FREQUENCY") || fileString.Contains("//"))
                        {
                            // Skip Header or comment line
                        }
                        else
                        {
                            // Read the cal data
                            string[] columns = fileString.Split(',');
                            cellFreqs[linesRead] = Convert.ToDouble(columns[0]);
                            linesRead++;
                        }
                    }
                    cellDataRead = true;
                }
                if (fileString.Contains("WLAN DATA"))
                {
                    int linesRead = 0;
                    while (linesRead < numWlanPoints)
                    {
                        fileString = rd.ReadLine();
                        fileString = fileString.ToUpper();
                        if (fileString.Contains("NUMHARMS"))
                        {
                            // Read the cal data
                            string[] columns = fileString.Split(',');
                            numWlanHarms = Convert.ToInt32(columns[1]);
                            continue;
                        }
                        if (fileString.Contains("FREQUENCY") || fileString.Contains("//"))
                        {
                            // Skip Header or comment line
                        }
                        else
                        {
                            // Read the cal data
                            string[] columns = fileString.Split(',');
                            wlanFreqs[linesRead] = Convert.ToDouble(columns[0]);
                            linesRead++;
                        }
                    }
                    wlanDataRead = true;
                }
            }
            rd.Close();
            rs.Close();
        }
    }

}
